<?php

namespace Core;

/**
 * Class request to handle all requests going through application
 */

class Request
{
    /**
     * Request URI
     *
     * @var string
     */

    private $URL;

    /**
     * Base URL
     *
     * @var string
     */

    private $baseURL;

    private $files = [];

    /**
     * Set Up Request and Base URLS
     *
     * @return void
     */

    public function setURLs()
    {

        if ($_SERVER['DOCUMENT_ROOT'] == dirname($_SERVER['SCRIPT_FILENAME'])) {

            $basePath = '/';
        } else {

            $basePath = str_replace('/public', '', dirname($_SERVER['SCRIPT_NAME']));
        }

        //Set base url (ex. http://site.com)

        $this->baseURL = $this->server('REQUEST_SCHEME').'://'.$this->server('HTTP_HOST').$basePath;

        //Get the current requested URL

        $requestedURI = $this->server('REQUEST_URI');

        //If there is any query in the url remove it (ex. http://site.com/index?id=23)

        if (strpos($requestedURI, '?') !== false) {

            list($requestedURI, $q) = explode('?', $requestedURI);
        }

        //First remove last '/' from $requestedURI (ex. http://site.com/list/  => http://site.com/list) so these requests will be the same

        //Then strip the script name from the url so we can match it to any of the (Fixed Routes @public/index) (ex. /books/list  => /list)

        $this->URL = rtrim(preg_replace('#^'.'/'.APP_DIR.'#', '', $requestedURI), '/');

        //If $requestedURI is empty (ex: homepage = http://site.com/scriptname) let $requestedURI = '/'

        $this->URL = $this->URL === '' ? '/' : $this->URL;
    }

    /**
     * get $_SERVER value by the given $key
     *
     * @param string $key
     * @var mixed $default
     * @return mixed
     */

    public function server($key, $default = null)
    {

        return array_key_exists($key, $_SERVER) ? $_SERVER[$key] : $default;
    }

    /**
     * get $_POST value by the given $key
     *
     * @param string $key
     * @var mixed $default
     * @return mixed
     */

    public function post($key, $default = null)
    {

        return array_key_exists($key, $_POST) ? $_POST[$key] : $default;
    }

    /**
     * get value from _GET by the given $key
     *
     * @param string $key
     * @param mixed $default
     * @return mixed
     */

    public function get($key, $default = null)
    {

        return array_key_exists($key, $_GET) ? $_GET[$key] : $default;
    }

    /**
     * Get the current requested URL
     *
     * @return string
     */

    public function getUrl()
    {

        return $this->URL;
    }

    /**
     * Get base url (ex. http://site.com)
     *
     * @return string
     */

    public function getBaseUrl()
    {

        return $this->baseURL;
    }

    /**
     * Get the current method used in the request
     *
     * @return string
     */

    public function getMethod()
    {

        return $this->server('REQUEST_METHOD');
    }

    public function getIP()
    {

        if (! is_null($this->server('HTTP_CLIENT_IP'))) {

            $ip = $this->server('HTTP_CLIENT_IP');
        } elseif (! is_null($this->server('HTTP_X_FORWARDED_FOR'))) {

            $ip = $this->server('HTTP_X_FORWARDED_FOR');
        } else {

            $ip = $this->server('REMOTE_ADDR');
        }

        return $ip;
    }

    public function file($input)
    {

        if (isset($this->files[$input])) {

            return $this->files[$input];
        }

        $uploadedFile = new UploadedFile($input);

        $this->files[$input] = $uploadedFile;

        return $this->files[$input];
    }
}